﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data;
using DataAccess;
/// <summary>
/// Pathway event methods and properties
/// </summary>
public class CPWEvent
{
    /// <summary>
    /// Default constructor
    /// </summary>
	public CPWEvent()
	{
		//
		// TODO: Add constructor logic here
		//
	}

    /// <summary>
    /// get recent note titles used by events
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetRecentClinicDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PWEVENT.GetRecentClinicRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }


    /// <summary>
    /// get recent note titles used by events
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetRecentNoteTitleDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PWEVENT.GetRecentNoteTitleRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }


    /// <summary>
    /// add a module to an event
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="lMID"></param>
    /// <param name="lPatPWEventMID"></param>
    /// <returns></returns>
    public bool InsertPWEventModule(BaseMaster BaseMstr,
                               string strPatientID,
                               long lPatPWEventID,
                               long lMID,
                               out long lPatPWEventMID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";
        lPatPWEventMID = 0;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);
        plist.AddInputParameter("pi_nMID", lMID);
        plist.AddOutputParameter("po_nEventModuleID", lPatPWEventMID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.InsertPatPWEventModule",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            CDataParameter paramValue2 = plist.GetItemByName("po_nEventModuleID");
            lPatPWEventMID = paramValue2.LongParameterValue;

            return true;
        }

        return false;
    }

    /// <summary>
    /// insert a new pathway event
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lPatPWID"></param>
    /// <param name="lRegionID"></param>
    /// <param name="lSiteID"></param>
    /// <param name="strEventTitle"></param>
    /// <param name="dtEventDate"></param>
    /// <param name="lPatPWEventID"></param>
    /// <returns></returns>
    public bool InsertPWEvent(BaseMaster BaseMstr,
                               string strPatientID,
                               long lPatPWID,
                               long lRegionID,
                               long lSiteID,
                               string strEventTitle,
                               DateTime dtEventDate,
                               out long lPatPWEventID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";
        lPatPWEventID = 0;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", lRegionID);
        plist.AddInputParameter("pi_nSiteID", lSiteID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nPatPWID", lPatPWID);

        plist.AddInputParameter("pi_vEventTitle", strEventTitle);
        plist.AddInputParameter("pi_dtEventDate", dtEventDate);

        plist.AddOutputParameter("po_nPatPWEventID", lPatPWEventID);


        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.InsertPatPWEvent",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            CDataParameter paramValue2 = plist.GetItemByName("po_nPatPWEventID");
            lPatPWEventID = paramValue2.LongParameterValue;
           
            return true;
        }

        return false;
    }

    /// <summary>
    /// get todays events
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lPatPWID"></param>
    /// <returns></returns>
    public DataSet GetTodaysPatPWEventDS(BaseMaster BaseMstr,
                                         string strPatientID,
                                         long lPatPWID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nPatPWID", lPatPWID);


        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PWEVENT.GetTodaysPatPWEventRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6942 get list of event entry types Subjective, Objective, Assessment, Plan etc...
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetNoPathwayDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetNoPathwayRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }


    /// <summary>
    /// US:6942 get list of event entry types Subjective, Objective, Assessment, Plan etc...
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetOpenEncountersDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetOpenEncountersRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// Updates the intrument date
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="dtDate"></param>
    /// <returns></returns>
    public bool UpdateInstrumentDate(BaseMaster BaseMstr,
                                     long lIntakeID,
                                     DateTime dtDate)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_dtDate", dtDate);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.UpdateInstrumentDate",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// get the most patients recent medical history dataset
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <returns></returns>
    public DataSet GetMedicalHxDS(BaseMaster BaseMstr,
                                  string strPatientID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PWEVENT.GetMedicalHxRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }
    



    /// <summary>
    /// insert past medical history record
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWID"></param>
    /// <param name="lPatPWEventID"></param>
    /// <returns></returns>
    public bool InsertMedicalHx( BaseMaster BaseMstr,
                                 long lPatPWID,
                                 long lPatPWEventID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);
        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWID", lPatPWID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.InsertMedicalHx",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// get the most recent medical history intake_id
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <returns></returns>
    public long GetMedicalHxIntakeID(BaseMaster BaseMstr,
                                     string strPatientID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        long lIntakeID = 0;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        
        plist.AddOutputParameter("po_nIntakeID", lIntakeID);

        bool bStatus = BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.GetMedicalHxIntakeID",
                                                       plist,
                                                       out lStatusCode,
                                                       out strStatusComment);

        //get out params
        CDataParameter paramValue1 = plist.GetItemByName("po_nIntakeID");
        lIntakeID = paramValue1.LongParameterValue;

        if (lStatusCode != 0)
        {
            BaseMstr.StatusCode = 1;
            BaseMstr.StatusComment = "Unable to get latest Medical History record.";
        }

        return lIntakeID;
    }

    /// <summary>
    /// US:7109 write the tiu note to TBI after written to MDWS
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="strDFN"></param>
    /// <param name="strDUZ"></param>
    /// <param name="strEncounterID"></param>
    /// <param name="strConsultIEN"></param>
    /// <param name="strCosignerDUZ"></param>
    /// <param name="strNoteTitleIEN"></param>
    /// <param name="strTIUNote"></param>
    /// <param name="lClinicID"></param>
    /// <param name="lEncounterType"></param>
    /// <param name="strNoteTitle"></param>
    /// <param name="strLocation"></param>
    /// <param name="strCosigner"></param>
    /// <returns></returns>
    public bool WriteNote(BaseMaster BaseMstr,
                          long lPatPWEventID,
                          string strDFN,
                          string strDUZ,
                          string strEncounterID,
                          string strConsultIEN,
                          string strCosignerDUZ,
                          string strNoteTitleIEN,
                          string strTIUNoteID,
                          string strTIUNote,
                          long lClinicID,
                          long lEncounterType,
                          string strNoteTitle,
                          string strLocation,
                          string strCosigner,
                          string strConsult,
                          string strTBINote)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);
        plist.AddInputParameter("pi_vDFN", strDFN); 
        plist.AddInputParameter("pi_vDUZ", strDUZ);
        plist.AddInputParameter("pi_vEncounterID", strEncounterID); 
        plist.AddInputParameter("pi_vConsultIEN", strConsultIEN); 
        plist.AddInputParameter("pi_vCosignerDUZ", strCosignerDUZ); 
        plist.AddInputParameter("pi_vNoteTitleIEN", strNoteTitleIEN);
        plist.AddInputParameter("pi_vTIUNoteID", strTIUNoteID);
        plist.AddInputParameterCLOB("pi_vTIUNote", strTIUNote); 
        plist.AddInputParameter("pi_nClinicID", lClinicID); 
        plist.AddInputParameter("pi_nEncounterType", lEncounterType);
        plist.AddInputParameter("pi_vNoteTitle", strNoteTitle);  
        plist.AddInputParameter("pi_vLocation", strLocation);
        plist.AddInputParameter("pi_vCosigner", strCosigner);
        plist.AddInputParameter("pi_vConsult", strConsult);
        plist.AddInputParameterCLOB("pi_vTBINote", strTBINote);
  
        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.WriteNote",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// US:7109 save note progress, used while the note is being written but before it is commited
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="lEncounterType"></param>
    /// <param name="strNoteTitle"></param>
    /// <param name="strLocation"></param>
    /// <param name="strCosigner"></param>
    /// <param name="strConsult"></param>
    /// <param name="strNote"></param>
    /// <returns></returns>
    public bool SaveNoteProgress(BaseMaster BaseMstr,
                                  long lPatPWEventID,
                                  long lEncounterType,
                                  string strNoteTitle,
                                  string strLocation,
                                  string strCosigner,
                                  string strConsult,
                                  string strNote)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);
        plist.AddInputParameter("pi_nEncounterType", lEncounterType);
        plist.AddInputParameter("pi_vNoteTitle", strNoteTitle);
        plist.AddInputParameter("pi_vLocation", strLocation);
        plist.AddInputParameter("pi_vCosigner", strCosigner);
        plist.AddInputParameter("pi_vConsult", strConsult);
        plist.AddInputParameterCLOB("pi_vNote", strNote);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.SaveNoteProgress",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }


    /// <summary>
    /// US:7003 US:7002 US:7001 Gets next pathway event associated with a patient
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lPatPWID"></param>
    /// <returns></returns>
    public DataSet GetNextPatPWEventDS(BaseMaster BaseMstr,
                                       string strPatientID,
                                       long lPatPWEventID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PWEVENT.GetNextPatPWEventRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }
    

    /// <summary>
    /// US:6831 US:6832 US:6833 US:6834 Get outcome measures fro a pathway event
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <returns></returns>
    public DataSet GetOutcomeMeasureDS(BaseMaster BaseMstr,
                                       long lPatPWEventID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetOutcomeMeasuresRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6942 US:6831 US:6832 US:6833 US:6834 Get list of event entries for an event
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <returns></returns>
    public DataSet GetEventEntryDS(BaseMaster BaseMstr,
                                   long lPatPWEventID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetEventEntryRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6831 US:6832 US:6833 US:6834 Get all scores for an intake
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public DataSet GetIntakeScoreDS(BaseMaster BaseMstr,
                                    long lIntakeID,
                                    long lMID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nIntakeID", lIntakeID);
        plist.AddInputParameter("pi_nMID", lMID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetIntakeScoreRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }
      

    /// <summary>
    /// US:6942 get list of event entry types Subjective, Objective, Assessment, Plan etc...
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetEventEntryTypeDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetEventEntryTypeRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6942 get list of event statuses Better, Same, Worse etc...
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetEventStatusDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetEventStatusRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// US:6942 get list of event severities unk, mild, moderate, severe
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <returns></returns>
    public DataSet GetEventSeverityDS(BaseMaster BaseMstr)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                          "PCK_PWEVENT.GetEventSeverityRS",
                                           plist,
                                           out lStatusCode,
                                           out strStatusComment);

        //set the base master status code and status for display
        BaseMstr.StatusCode = lStatusCode;
        BaseMstr.StatusComment = strStatusComment;

        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }


    /// <summary>
    /// US:6942 insert an event entry 
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="lPWEventModuleID"></param>
    /// <param name="lEventEntryTypeID"></param>
    /// <param name="strEventEntryComment"></param>
    /// <param name="lEventStatusID"></param>
    /// <returns></returns>
    public long InsertPWEventEntry(BaseMaster BaseMstr,
                                   long lPatPWEventID,
                                   long lPWEventModuleID,
                                   long lEventEntryTypeID,
                                   DateTime dtEntryDateTime,
                                   string strEventEntryComment,
                                   long lEventStatusID,
                                   long lEventSeverityID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";
        long lEventEntryID = 0;


        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", BaseMstr.SelectedPatientID);
        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);
        plist.AddInputParameter("pi_nPWEventModuleID", lPWEventModuleID);
        plist.AddInputParameter("pi_nEventEntryTypeID", lEventEntryTypeID);
        plist.AddInputParameter("pi_dtEventDate", dtEntryDateTime);
        plist.AddInputParameter("pi_vEventEntryComment", strEventEntryComment);
        plist.AddInputParameter("pi_nEventStatusID", lEventStatusID);
        plist.AddInputParameter("pi_nEventSeverityID", lEventSeverityID);

        plist.AddOutputParameter("po_nEventEntryID", lEventEntryID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.InsertPWEventEntry",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            //get the returned patient pathway id
            CDataParameter param = plist.GetItemByName("po_nEventEntryID");
            lEventEntryID = param.LongParameterValue;
        }

        return lEventEntryID;
    }

    /// <summary>
    /// US:7003 insert a new patient pathway event 
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lRegionID"></param>
    /// <param name="lSiteID"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lPatPWID"></param>
    /// <param name="strEventTitle"></param>
    /// <param name="dtEventDate"></param>
    /// <param name="lRelatedPatPWID"></param>
    /// <returns></returns>
    public long InsertPWEvent(BaseMaster BaseMstr,
                               long lRegionID,
                               long lSiteID,
                               string strPatientID,
                               long lPatPWID,
                               string strEventTitle,
                               DateTime dtEventDate,
                               long lRelatedPatPWID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";
        
        long lPatPWEventID = 0;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", lRegionID);
        plist.AddInputParameter("pi_nSiteID", lSiteID);
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nPatPWID", lPatPWID);
        plist.AddInputParameter("pi_vEventTitle", strEventTitle);
        plist.AddInputParameter("pi_dtEventDate", dtEventDate);
        plist.AddInputParameter("pi_nRelatedEventID", lRelatedPatPWID);
        plist.AddOutputParameter("po_nPatPWEventID", lPatPWEventID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.InsertPWEvent",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            //get the returned patient pathway id
            CDataParameter param = plist.GetItemByName("po_nPatPWEventID");
            lPatPWEventID = param.LongParameterValue;
        }

        return lPatPWEventID;
    }

    /// <summary>
    /// US:6942 update an event entry 
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="lPWEventModuleID"></param>
    /// <param name="lEventEntryTypeID"></param>
    /// <param name="strEventEntryComment"></param>
    /// <param name="lEventStatusID"></param>
    /// <returns></returns>
    public bool UpdatePWEventEntry(BaseMaster BaseMstr,
                                   long lPatPWEventEntryID,
                                   long lPWEventModuleID,
                                   long lEventEntryTypeID,
                                   DateTime dtEventDate,
                                   string strEventEntryComment,
                                   long lEventStatusID,
                                   long lEventSeverityID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";
      

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nEventEntryID", lPatPWEventEntryID);
        plist.AddInputParameter("pi_nPWEventModuleID", lPWEventModuleID);
        plist.AddInputParameter("pi_dtEventDate", dtEventDate);

        plist.AddInputParameter("pi_vEventEntryComment", strEventEntryComment);
        plist.AddInputParameter("pi_nEventStatusID", lEventStatusID);
        plist.AddInputParameter("pi_nEventSeverityID", lEventSeverityID);
      

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.UpdatePWEventEntry",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// US:7003 US:7002 US:7001 update an events plan
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventID"></param>
    /// <param name="lPatPWPlanEventID"></param>
    /// <param name="strPlanText"></param>
    /// <returns></returns>
    public bool UpdatePWEventPlan(BaseMaster BaseMstr,
                                  long lPatPWEventID,
                                  long lPatPWPlanEventID,
                                  string strPlanText)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";


        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nPatPWEventID", lPatPWEventID);
        plist.AddInputParameter("pi_nPatPWPlanEventID", lPatPWPlanEventID);
        plist.AddCLOBParameter("pi_vPlanText", strPlanText, ParameterDirection.Input);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.UpdatePWEventPlan",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// US:6942 delete an event entry
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="lPatPWEventEntryID"></param>
    /// <returns></returns>
    public bool DeletePWEventEntry(BaseMaster BaseMstr,
                                   long lPatPWEventEntryID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";


        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nEventEntryID", lPatPWEventEntryID);
        
        BaseMstr.DBConn.ExecuteOracleSP("PCK_PWEVENT.DeletePWEventEntry",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }
        
}